define(['angular', 'app'], function (angular, app) {
	"use strict";

	app.service('AbstractHealthMoodService', function ($parse, $q, formatter, CollectionCRUDService,
        localResourceDirectoryService,modalService, $filter) {



		var AbstractHealthMoodService = function (extension) {
			var config = {},
				dateKey = extension.object.dateKey;

			var deepObjectSet = function (keys, item, value) {
				var temp = item,
					keyArray = keys.split("."),
					numKeys = keyArray.length;

				for(var i = 0; i < numKeys - 1; i++) {
					if (!temp[keyArray[i]]) {
						temp[keyArray[i]] = {};
					}
					temp = temp[keyArray[i]];
				}
				temp[keyArray[numKeys - 1]] = value;
			};

			config.dependency = {
				service: localResourceDirectoryService
			};

			config.collection = {
				sortComparator: [ { key: dateKey[0].from + "SortKey", reverse: true } ],
				filter: function(list, filters) {
					if (filters === null) {
						return list;
					}
					
					var startDate = new Date(filters.startDate + " 00:00"),
						endDate = new Date(filters.endDate + " 23:59"),
						getter = $parse(dateKey[0].from);

					return list.filter(function(item){
						var itemDate = new Date(getter(item));
						return itemDate >= startDate && itemDate <= endDate;
					});
				}
			};

            var formatNumber = function(number, numDecimals) {
                return ($filter('number')(number, numDecimals)).replace(",", "");
            };

			config.object = {
				responseTransform: function (item) {
                    if(item.name === 'Weight' && item.valueQuantity && item.valueQuantity.value) {
                        item.valueQuantity.value = formatNumber(item.valueQuantity.value, 1);
                    }

					if (!item.method && item.name === 'Glucose') {
						item.method = 'Unknown'
					}

					dateKey.forEach(function(key){
						var dateOnlyKey = key.toPrefix ? key.toPrefix + "Date" : "inputEntryDate",
							timeOnlyKey = key.toPrefix ? key.toPrefix + "Time" : "inputEntryTime",
							value = $parse(key.from)(item),
							dateOnlyValue = formatter.getFormattedFrontendDate(value),
							timeOnlyValue = formatter.getFormattedFrontendTime(value);

						deepObjectSet(dateOnlyKey, item, dateOnlyValue);
						deepObjectSet(timeOnlyKey, item, timeOnlyValue);
						deepObjectSet(key.from, item, dateOnlyValue + " " + timeOnlyValue);

						if (dateOnlyValue && timeOnlyValue) {
							deepObjectSet(key.from + "SortKey", item, new Date(dateOnlyValue + " " + timeOnlyValue));
						} else {
							deepObjectSet(key.from + "SortKey", item, "");
						}
					});

					// Following transform guarantees array ordering of Systolic, Diastolic, and Pulse
					if (item.name === "BloodPressureAndPulse") {
						var orderedEntries,
							sys = _.find(item.entry, {name: '8480-6'}),
							dia = _.find(item.entry, {name: '8462-4'}),
							bpm = _.find(item.entry, {name: '8867-4'});

						orderedEntries = [sys, dia, bpm];
						delete item.entry;
						item.entry = orderedEntries;
					}

					// Following transform guarantees array ordering of Cholesterol
					if (item.name === "Cholesterol") {
						var orderedEntries,
							ldl = _.find(item.entry, {name: 'LDL'})

						orderedEntries = [ldl];
						delete item.entry;
						item.entry = orderedEntries;
					}

					//add flag to indicate abnormal BP/Pulse reading
					if (item.name==="BloodPressureAndPulse"){						
						item.entry.forEach(function(itm){
							if (itm && itm.valueQuantity.value && itm.valueQuantity.value.length>0) {
								if ((itm.name=="8480-6" && (itm.valueQuantity.value>180 || itm.valueQuantity.value <80)) ||							
									(itm.name=="8462-4"  && (itm.valueQuantity.value>110 || itm.valueQuantity.value <40)) ||
									(itm.name=="8867-4" && (itm.valueQuantity.value>150 || itm.valueQuantity.value <50))) {
										itm["outOfBounds"] = true;								
								}
							}
						});						
					}

					//add flag to indicate abnormal glucose reading
					if (item.name==="Glucose") {
						if (item.valueQuantity.value>300 || item.valueQuantity.value<70) {
							item['outOfBounds'] = true;
						}
					}
				

					return item;
				},
				requestTransform: function (item) {
					dateKey.forEach(function(key) {
						var dateOnlyKey = key.toPrefix ? key.toPrefix + "Date" : "inputEntryDate",
							timeOnlyKey = key.toPrefix ? key.toPrefix + "Time" : "inputEntryTime";
						deepObjectSet(key.from, item, formatter.getFormattedBackendDateTime($parse(dateOnlyKey)(item) + " " + $parse(timeOnlyKey)(item)));
					});

					!item.notes && delete item.notes;

					return item;
				}
			};

			angular.element.extend(true, config, extension);

			CollectionCRUDService.call(this, config);

			this.createEmpty = function () {
				var empty = { notes: '' };

				extension.object.dateKey.forEach(function(key) {
					deepObjectSet(key.from, empty, formatter.getFormattedBackendDateTime(new Date()));
				});
			
                return this.config.object.responseTransform(angular.extend(empty, angular.copy(extension.object.defaults)));
			};

			this.fetch = function (queryParams) {
				var that = this,
					requestPromise = that._onRequestComplete(that.BaseCRUDService.fetch(queryParams));

				requestPromise.then(function (response) {
					that.dataPointers.filters = angular.copy(queryParams);
					that.dataPointers.list = sort(response[that.config.collection.name], that.config.collection.sortComparator);
					that.dataPointers.unfilteredList = angular.copy(that.dataPointers.list);
					that.dataPointers.list = that.config.collection.filter(angular.copy(that.dataPointers.list), angular.copy(that.dataPointers.filters));
					that.dataPointers.link = response.link;
				});
				return requestPromise;
			};

            this.save = function (item) {

				var warn = false;

				//test if warning required for BP/Pulse entry
				if (item.name=="BloodPressureAndPulse") {
					if (item.entry) {						
						for (var i=0; i<item.entry.length; i++) {
							if (item.entry[i].valueQuantity.value) { 
								var intVal = parseInt(item.entry[i].valueQuantity.value);
								if (!isNaN(intVal)) {
									if ((item.entry[i].name=="8480-6" && (intVal > 180 || intVal < 80)) ||							
										(item.entry[i].name=="8462-4" && (intVal > 110 || intVal < 40)) ||
										(item.entry[i].name=="8867-4" && (intVal > 150 || intVal < 50)))
									warn = true;
								}
							}
						}						
					}
				}

				//test if warning required for a glucose entry
				if (item.name==="Glucose") {
					if (item.valueQuantity.value && !isNaN(parseInt(item.valueQuantity.value))) {
						if (item.valueQuantity.value>300 || item.valueQuantity.value<70) {
							warn = true;
						}
					}
				}

				if (warn) {
					var modalOptions = {
						closeButtonText: 'Continue',
						headerText: "Warning",
						bodyText:  "The value(s) you entered are outside the normal range. Talk to your health care provider about what is OK for you. If you feel sick and need urgent help, please Call 911.",
						type : "danger"
					};
					modalService.showModal({}, modalOptions).then(function() {});
				}

                var that = this,
                    requestPromise = that._onRequestComplete(that.BaseCRUDService.save(angular.copy(item)));

                requestPromise.then(function (responseItem) {
                    if(that.dataPointers.list && extension.collection.objectType !== "HealthTrackerPreferences") {
    					that._deleteFromLocalList(item);
    					var scopedList = angular.copy(that.dataPointers.list);
    					scopedList.push(responseItem);
    					scopedList = that.config.collection.filter ? that.config.collection.filter(scopedList, that.dataPointers.filters) : scopedList;
    					that.dataPointers.list = sort(scopedList, that.config.collection.sortComparator);
    					if (!item.link && that.dataPointers.unfilteredItemCount !== null) {
    						that.dataPointers.unfilteredItemCount++;
    					}
    					that.latest = null;
    					if (that.dataPointers.list.length > 0)
    						that.latest = that.dataPointers.list[0];
    				} else if (that.config.object.supportsLatest && that.latest) {
    					that.latest = null;
    					that.fetchLatest();
    				}

                    if(that.dataPointers.unfilteredList) {
                        that.deleteFromUnfilteredList(item);
                        var scopedList = angular.copy(that.dataPointers.unfilteredList);
                        scopedList.push(responseItem);
                        that.dataPointers.unfilteredList = scopedList;
                    }
                });

                return requestPromise;
            };

            this.delete = function (item) {
                var that = this,
                    requestPromise = that.BaseCRUDService.delete(item);

                requestPromise.then(function () {
                    that._deleteFromLocalList(item);
                    that.deleteFromUnfilteredList(item);

                    if (that.dataPointers.unfilteredItemCount !== null && that.dataPointers.unfilteredItemCount > 0) {
                        that.dataPointers.unfilteredItemCount--;
                    }
                    if (that.config.object.supportsLatest && that.latest && that.idGetter(item) == that.idGetter(that.latest)) {
                    	that.latest = null;
                    	that.fetchLatest();
                    }
                });

                return requestPromise;
            };

            this.deleteFromUnfilteredList = function(item) {
                var list = this.dataPointers.unfilteredList,
                    expectedId = this.idGetter(item);

                if (expectedId && list) {
                    var deleteIndex = list.indexOf(this.unfilteredListGetById(this.idGetter(item)));

                    if(deleteIndex !== -1) {
                        list.splice(deleteIndex, 1);
                    }
                }
            };

            this.unfilteredListGetById = function(id) {
                var list = this.dataPointers.unfilteredList,
                    listLength = list ? list.length : 0;

                for(var i = 0; i < listLength; i++) {
                    if(angular.equals(id, this.idGetter(list[i]))){
                        return list[i];
                    }
                }
            };

            this.getLatest = function () {
            	if (this.latest) {
            		var defer = $q.defer();
            		defer.resolve();
            		return defer.promise;
            	} else {
            		// TODO: maybe use the latest from dataPointers...
            		return this.fetchLatest();
            	}
            };
		};

		// ============================================
		// internal functions
		// ============================================
		 var sort = function (list, sortComparator) {
			return list.sort(function (left, right) {
				var resolvedOrder = 0;
				for (var i = 0; i < sortComparator.length; i++) {
					var getter = $parse(sortComparator[i].key),
						leftVal = getter(left),
						rightVal = getter(right);

					if (angular.isString(leftVal) && angular.isString(rightVal)) {
						leftVal = leftVal.toLowerCase();
						rightVal = rightVal.toLowerCase();
					}

					if (leftVal === rightVal) {
						resolvedOrder = 0;
					} else {
						if (angular.isUndefined(leftVal) && angular.isDefined(rightVal)) {
							resolvedOrder = -1;
						} else if (angular.isDefined(leftVal) && angular.isUndefined(rightVal)) {
							resolvedOrder = 1;
						} else {
							resolvedOrder = leftVal > rightVal ? 1 : -1;
							resolvedOrder = sortComparator[i].reverse ? -1 * resolvedOrder : resolvedOrder;
						}
						break;
					}
				}
				return resolvedOrder;
			});
		};

		AbstractHealthMoodService.prototype = Object.create(CollectionCRUDService.prototype);
		AbstractHealthMoodService.prototype.constructor = AbstractHealthMoodService;

		return AbstractHealthMoodService;
	});
});